/* $Id: kqemu-freebsd.c,v 1.6 2006/04/25 22:16:42 bellard Exp $ */
/* $Id:$ port to DragonFly by Naoya Sugioka with many helps from Hofmann Johannes and others */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/ctype.h>
#include <sys/fcntl.h>
#include <sys/ioccom.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/proc.h>
#include <sys/resourcevar.h>
#include <sys/sched.h>
#include <sys/signal2.h>
#include <sys/kernel.h>
#include <sys/sysctl.h>
#include <sys/device.h>
#include <sys/eventhandler.h>

#include <vm/vm.h>
#include <vm/vm_param.h>
#include <vm/vm_extern.h>
#include <vm/pmap.h>
#include <vm/vm_map.h>
#include <vm/vm_kern.h>
#include <vm/vm_page.h>

#include <machine/vmparam.h>
#include <machine/stdarg.h>

#include "kqemu-kernel.h"

#ifndef KQEMU_MAJOR
#define KQEMU_MAJOR 250
#endif

MALLOC_DECLARE(M_KQEMU);
MALLOC_DEFINE(M_KQEMU, "kqemu", "kqemu buffers");

int kqemu_debug;
SYSCTL_INT(_debug, OID_AUTO, kqemu_debug, CTLFLAG_RW, &kqemu_debug, 0,
       "kqemu debug flag");

#define	USER_BASE	0x1000

#if defined(__i386__)
#define BTOP(x)  i386_btop(x)
#elif defined(__amd64__)
#define BTOP(x)  amd64_btop(x)
#endif

/* lock the page at virtual address 'user_addr' and return its
   physical page index. Return NULL if error */
struct kqemu_user_page *CDECL kqemu_lock_user_page(unsigned long *ppage_index,
                                                   unsigned long user_addr)
{
    struct vmspace *vm = curproc->p_vmspace;
    vm_offset_t va = user_addr;
    vm_paddr_t pa = 0;
    int ret;
    pmap_t pmap;
    
    ret = vm_map_wire(&vm->vm_map, va, va+PAGE_SIZE, 0);
    if (ret != KERN_SUCCESS) {
	kqemu_log("kqemu_lock_user_page(%08lx) failed, ret=%d\n", va, ret);
	return NULL;
    }
    pmap = vm_map_pmap(&vm->vm_map);
    pa = pmap_extract(pmap, va);
    /* kqemu_log("kqemu_lock_user_page(%08lx) va=%08x pa=%08x\n", user_addr, va, pa); */
    *ppage_index = BTOP(pa);
    return (struct kqemu_user_page *)va;
}

void CDECL kqemu_unlock_user_page(struct kqemu_user_page *page)
{
    struct vmspace *vm = curproc->p_vmspace;
    vm_offset_t va;
    int ret;
    /* kqemu_log("kqemu_unlock_user_page(%08lx)\n", page_index); */
    va = (vm_offset_t)page;

    ret = vm_map_wire(&vm->vm_map, va, va+PAGE_SIZE, KM_PAGEABLE);
    if (ret != KERN_SUCCESS) {
	kqemu_log("kqemu_unlock_user_page(%08lx) failed, ret=%d\n", va, ret);
    }
}

/*
 * Allocate a new page. The page must be mapped in the kernel space.
 * Return the page_index or -1 if error.
 */
struct kqemu_page *CDECL kqemu_alloc_zeroed_page(unsigned long *ppage_index)
{
    pmap_t pmap;
    vm_offset_t va;
    vm_paddr_t pa;

    va = kmem_alloc(&kernel_map, PAGE_SIZE);
    if (va == 0) {
	kqemu_log("kqemu_alloc_zeroed_page: NULL\n");
	return NULL;
    }
    pmap = vm_map_pmap(&kernel_map);
    pa = pmap_extract(pmap, va);
    /* kqemu_log("kqemu_alloc_zeroed_page: %08x\n", pa); */
    *ppage_index = BTOP(pa);
    return (struct kqemu_page *)va;
}

void CDECL kqemu_free_page(struct kqemu_page *page)
{
    if (kqemu_debug > 0)
    	kqemu_log("kqemu_free_page(%p)\n", page);
    kmem_free(&kernel_map, (vm_offset_t) page, PAGE_SIZE);
}

/* return kernel address of the physical page page_index */
void * CDECL kqemu_page_kaddr(struct kqemu_page *page)
{
    vm_offset_t va = (vm_offset_t)page;
    return (void *)va;
}

/* contraint: each page of the vmalloced area must be in the first 4
   GB of physical memory */
void * CDECL kqemu_vmalloc(unsigned int size)
{
    void *ptr = kmalloc(size, M_KQEMU, M_WAITOK);
    if (kqemu_debug > 0)
	kqemu_log("kqemu_vmalloc(%d): %p\n", size, ptr);
    return ptr;
}

void CDECL kqemu_vfree(void *ptr)
{
    if (kqemu_debug > 0)
	kqemu_log("kqemu_vfree(%p)\n", ptr);
    kfree(ptr, M_KQEMU);
}

/* return the physical page index for a given virtual page */
unsigned long CDECL kqemu_vmalloc_to_phys(const void *vaddr)
{
    vm_paddr_t pa = vtophys(vaddr);
    if (pa == 0) {
	kqemu_log("kqemu_vmalloc_to_phys(%p)->error\n", vaddr);
	return -1;
    }
    if (kqemu_debug > 0)
	kqemu_log("kqemu_vmalloc_to_phys(%p)->%08x\n", vaddr, pa);
    return BTOP(pa);
}

/* Map a IO area in the kernel address space and return its
   address. Return NULL if error or not implemented.  */
void * CDECL kqemu_io_map(unsigned long page_index, unsigned int size)
{
    return NULL;
}

/* Unmap the IO area */
void CDECL kqemu_io_unmap(void *ptr, unsigned int size)
{
}

/* return TRUE if a signal is pending (i.e. the guest must stop
   execution) */
int CDECL kqemu_schedule(void)
{
    if (kqemu_debug > 0)
        kqemu_log("kqemu_schedule\n");

    lwkt_yield();
    return CURSIG(FIRST_LWP_IN_PROC(curproc));
}

static char log_buf[4096];

void CDECL kqemu_log(const char *fmt, ...)
{
    __va_list ap;
    __va_start(ap, fmt);
    kvsnprintf(log_buf, sizeof(log_buf), fmt, ap);
    kprintf("kqemu: %s", log_buf);
    __va_end(ap);
}

struct kqemu_instance { 
    TAILQ_ENTRY(kqemu_instance) kqemu_ent;
    struct cdev *kqemu_dev;
    /*    struct semaphore sem;  */
    struct kqemu_state *state;
};

static int kqemu_ref_count = 0;
static struct kqemu_global_state *kqemu_gs = NULL;

cdev_t kqemu_dev;

static d_close_t kqemu_close;
static d_open_t kqemu_open;
static d_ioctl_t kqemu_ioctl;

static struct dev_ops kqemu_ops = {
	{ "kqemu", KQEMU_MAJOR, 0},
	.d_open =	kqemu_open,
	.d_ioctl =	kqemu_ioctl,
	.d_close =	kqemu_close,
};

static void kqemu_destroy(struct kqemu_instance *ks)
{
    if (ks->state) {
        kqemu_delete(ks->state);
        ks->state = NULL;
    }

    kfree(ks, M_KQEMU);
    --kqemu_ref_count;
}

/* ARGSUSED */
static int
kqemu_open(struct dev_open_args *ap)
{
    cdev_t dev = ap->a_head.a_dev;
    int flags = ap->a_oflags;
    uid_t uid = ap->a_cred->cr_uid;
    struct kqemu_instance *ks;

    if ((flags & (FREAD|FWRITE)) == FREAD)
	return(EPERM);

    ks = kmalloc(sizeof(struct kqemu_instance), M_KQEMU, M_WAITOK);
    if (ks == NULL) {
	kqemu_log("malloc failed\n");
	return ENOMEM;
    }
    memset(ks, 0, sizeof *ks);
    kqemu_ref_count++;

    dev->si_drv1 = ks;
    if (kqemu_debug > 0)
	kqemu_log("opened by pid=%d\n", uid);
    return 0;
}

/* ARGSUSED */
static int
kqemu_ioctl(struct dev_ioctl_args *ap)
{
    cdev_t dev = ap->a_head.a_dev;
    u_long cmd = ap->a_cmd;
    caddr_t addr = ap->a_data;
    uid_t uid = ap->a_cred->cr_uid;
    int error = 0, ret;
    struct kqemu_instance *ks = dev->si_drv1;
    struct kqemu_state *s = ks->state;

    switch(cmd) {
    case KQEMU_INIT: {
	struct kqemu_init d1, *d = &d1;
	if (s != NULL) {
	    error = EIO;
	    break;
	}
	d1 = *(struct kqemu_init *)addr;
	if (kqemu_debug > 0)
	    kqemu_log("ram_base=%p ram_size=%ld\n", d1.ram_base, d1.ram_size);
	s = kqemu_init(d, kqemu_gs);
	if (s == NULL) {
	    error = ENOMEM;
	    break;
	}
	ks->state = s;
	break;
    }
    case KQEMU_SET_PHYS_MEM: {
        struct kqemu_phys_mem m1, *m = &m1;
        if (s == NULL) {
            error = EIO;
            break;
        }
	m1 = *(struct kqemu_phys_mem *)addr;
        error = kqemu_set_phys_mem(s, m);
        if (error != 0) {
            error = ENOMEM;
        }
        break;
    }
    case KQEMU_EXEC: {
	struct kqemu_cpu_state *ctx;
	if (s == NULL) {
	    error = EIO;
	    break;
	}
	ctx = kqemu_get_cpu_state(s);
	*ctx = *(struct kqemu_cpu_state *)addr;
	ret = kqemu_exec(s);
//	p->p_retval[0] = ret;
	*(struct kqemu_cpu_state *)addr = *ctx;
	break;
    }
    case KQEMU_GET_VERSION:
	*(int *)addr = KQEMU_VERSION;
	break;
    default:
	error = EINVAL;
    }
    return error;
}

/* ARGSUSED */
static int
kqemu_close(struct dev_close_args *ap)
{
    cdev_t dev = ap->a_head.a_dev;
    struct kqemu_instance *ks = (struct kqemu_instance *) dev->si_drv1;

    kqemu_destroy(ks);

    if (kqemu_debug > 0)
	kqemu_log("closed by pid=%d\n", 0);
    return 0;
}

/* ARGSUSED */
static int
kqemu_modevent(module_t mod __unused, int type, void *data __unused)
{
    int error = 0;
    int max_locked_pages;
    int rc;

    switch (type) {
    case MOD_LOAD:
	kprintf("kqemu version 0x%08x\n", KQEMU_VERSION);
	max_locked_pages = physmem / 2;
        kqemu_gs = kqemu_global_init(max_locked_pages);

	if ((rc = dev_ops_add(&kqemu_ops, 0, 0))) {
	    kqemu_log("error registering kqwemu_ops, rc=%d\n", rc);
            error = ENOENT;
            break;
	}
	kqemu_dev = make_dev(&kqemu_ops, 0,
			     UID_ROOT, GID_WHEEL, 0660, "kqemu");
	kqemu_log("KQEMU installed, max_locked_mem=%dkB.\n",
		  max_locked_pages * 4);

	/* We must obtain a reference to sdev, so we can destroy it later */
	reference_dev(kqemu_dev);

	kqemu_ref_count = 0;
	break;
    case MOD_UNLOAD:
	if (kqemu_ref_count > 0) {
            error = EBUSY;
            break;
        }

	destroy_dev(kqemu_dev);
	if ((rc = dev_ops_remove(&kqemu_ops, 0, 0)))
	    kqemu_log("error unregistering, rc=%d\n", rc);

        kqemu_global_delete(kqemu_gs);
        kqemu_gs = NULL;
	break;
    case MOD_SHUTDOWN:
	break;
    default:
	error = EOPNOTSUPP;
	break;
    }
    return (error);
}

DEV_MODULE(kqemu, kqemu_modevent, NULL);
MODULE_VERSION(kqemu, 1);
