/*
 * licensed as BSD 
 * Copyright Tobias Florek (2003)
 */


#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <fstab.h>


#define safe_me()							\
	/*seteuid( getuid() )*/

#define MOUNT_EXE "/sbin/mount"
	
int main( int, const char ** );	
void usage( void );
void printAvailable( void );
int usermount( const char * );
int hasopt( const char *, const char * );
void remopt( char *, const char * );

int
main( int argc, const char **argv )
{
	int err;

	if ( argc != 2 )
	{
		safe_me();
		usage();
		exit(1);
	}
	
	if ( argv[1][0] == '-' && argv[1][1] == 'p' )
	{
		safe_me();
		printAvailable();
		err = 0;
	}
	else
	{
		err = usermount( argv[1] );
	}
	exit( err );
}


void
printAvailable()
{
	struct fstab *fs;

	printf( "user-mountable devices:\n" );
	
	while ( (fs = getfsent()) != NULL )
	{
		if ( hasopt(fs->fs_mntops, "user") &&
				!hasopt(fs->fs_mntops, "nouser" ) )
			printf( "%s\t on %s\n", fs->fs_spec, fs->fs_file );
	}
}

int
usermount( const char *argv )
{
	struct fstab *fs;
	char *mntpoint;
	int err = 0;

	if ( ( (fs = getfsfile( argv )) != NULL ||
		       	 (fs = getfsspec( argv )) != NULL ) &&
			 hasopt( fs->fs_mntops, "user" ) &&
			!hasopt( fs->fs_mntops, "nouser" ) )
	{
		/* user is allowed to mount */
		remopt( fs->fs_mntops, "user" );
		remopt( fs->fs_mntops, "noauto" );

#ifdef MOUNT_IN_HOME
		/* if we want to mount with a different mountpoint: */
		mntpoint = malloc( sizeof('~') + sizeof( fs->fs_file ) );

		err = execl( MOUNT_EXE, "mount",
				"-o ", fs->fs_mntops, 
				"-t ", fs->fs_vfstype, 
				fs->fs_spec,
				mntpoint, 0 );

		free( mntpoint );
#else
		
		/* if we want to mount only on the mountpoint in /etc/fstab
		   this gets easier. */
		err = execl( MOUNT_EXE, "mount", fs->fs_file, 0 );
#endif
		
	}
	else
		fprintf( stderr, "Only root can do this.\n" );
	/* only reached, if error */
	return err;
}

int
hasopt( const char *mntopts, const char *option )
        
{
        int negative, found;
        char *opt, *optbuf;

        if (option[0] == 'n' && option[1] == 'o') {
                negative = 1;
                option += 2;
        } else
                negative = 0;
        optbuf = strdup(mntopts);
        found = 0;
        for (opt = optbuf; (opt = strtok(opt, ",")) != NULL; opt = NULL) {
                if (opt[0] == 'n' && opt[1] == 'o') {
                        if (!strcasecmp(opt + 2, option))
                                found = negative;
                } else if (!strcasecmp(opt, option))
                        found = !negative;
        }
        free(optbuf);
        return (found);
}


void
remopt(char *string, const char *opt)
{
        char *o, *p, *r;

        if (string == NULL || *string == '\0' || opt == NULL || *opt == '\0')
                return;

        r = string;

        for (p = string; (o = strsep(&p, ",")) != NULL;) {
                if (strcmp(opt, o) != 0) {
                        if (*r == ',' && *o != '\0')
                                r++;
                        while ((*r++ = *o++) != '\0')
                            ;
                        *--r = ',';
                }
        }
        *r = '\0';
}


void
usage( void )
{
	fprintf( stderr,  "%s\n",
			"usage: usermount mountpoint" );
}
